/**
 * OWASP AppSensor
 * 
 * This file is part of the Open Web Application Security Project (OWASP)
 * AppSensor project. For details, please see
 * <a href="http://www.owasp.org/index.php/Category:OWASP_AppSensor_Project">
 * 	http://www.owasp.org/index.php/Category:OWASP_AppSensor_Project</a>.
 *
 * Copyright (c) 2010 - The OWASP Foundation
 * 
 * AppSensor is published by OWASP under the BSD license. You should read and accept the
 * LICENSE before you use, modify, and/or redistribute this software.
 * 
 * @author Michael Coates <a href="http://www.aspectsecurity.com">Aspect Security</a>
 * @author John Melton <a href="http://www.jtmelton.com/">jtmelton</a>
 * @created 2010
 */
namespace org.owasp.appsensor.trendmonitoring.reference
{

    using System;
    using System.Collections.Generic;
    using org.owasp.appsensor.trendmonitoring;

    /**
     * This class is simply a data store for trend events.  There are a few accessor 
     * methods that allow access to the data in a few key ways. 
     * <p>
     * This data store is simply a proof of concept data store that allows the trend monitoring concepts
     * to be displayed.  While this data store will technically work in your application, 
     * no effort has been put into optimizing it for use.  It will continuously grow in size
     * as the application is used until restarted, meaning performance is likely poor.
     * 
     * @author Michael Coates (michael.coates .at. owasp.org) 
     *         <a href="http://www.aspectsecurity.com">Aspect Security</a>
     * @author John Melton (jtmelton .at. gmail.com)
     *         <a href="http://www.jtmelton.com/">jtmelton</a>
     * @since June 17, 2010
     */
    public class InMemoryTrendDataStore
    {
        private static List<TrendEvent> events = null;

        public void AddEvent(TrendEvent e)
        {
            InitializeEventsIfNecessary();
            events.Add(e);
        }

        public List<TrendEvent> GetCopyOfAllEvents()
        {
            InitializeEventsIfNecessary();
            List<TrendEvent> tempEvents = new List<TrendEvent>();
            tempEvents.AddRange(events);
            return tempEvents;
        }

        public Dictionary<String, List<TrendEvent>> GetCopyOfAllEventsByResource()
        {
            InitializeEventsIfNecessary();
            Dictionary<String, List<TrendEvent>> eventsByResource = new Dictionary<String, List<TrendEvent>>();

            //loop through all events
            foreach (TrendEvent te in events)
            {
                String resource = te.getResourceAccessed();
                //if resource is already in map
                if (eventsByResource.ContainsKey(resource))
                {
                    List<TrendEvent> tempEvents = eventsByResource[resource];
                    tempEvents.Add(te);
                    eventsByResource.Add(resource, tempEvents);
                }
                else
                {	//resource is not yet in map
                    List<TrendEvent> tempEvents = new List<TrendEvent>();
                    tempEvents.Add(te);
                    eventsByResource.Add(resource, tempEvents);
                }
            }

            return eventsByResource;
        }

        public Dictionary<String, List<TrendEvent>> GetCopyOfAllEventsByUserName()
        {
            InitializeEventsIfNecessary();
            Dictionary<String, List<TrendEvent>> eventsByUserName = new Dictionary<String, List<TrendEvent>>();

            //loop through all events
            foreach (TrendEvent te in events)
            {
                String userName = te.getAccessedByName();
                //if resource is already in map
                if (eventsByUserName.ContainsKey(userName))
                {
                    List<TrendEvent> tempEvents = eventsByUserName[userName];
                    tempEvents.Add(te);
                    eventsByUserName.Add(userName, tempEvents);
                }
                else
                {	//resource is not yet in map
                    List<TrendEvent> tempEvents = new List<TrendEvent>();
                    tempEvents.Add(te);
                    eventsByUserName.Add(userName, tempEvents);
                }
            }

            return eventsByUserName;
        }

        public Dictionary<String, List<TrendEvent>> GetCopyOfAllEventsByUserAddress()
        {
            InitializeEventsIfNecessary();
            Dictionary<String, List<TrendEvent>> eventsByUserAddress = new Dictionary<String, List<TrendEvent>>();

            //loop through all events
            foreach (TrendEvent te in events)
            {
                String userAddress = te.getAccessedByAddress();
                //if resource is already in map
                if (eventsByUserAddress.ContainsKey(userAddress))
                {
                    List<TrendEvent> tempEvents = eventsByUserAddress[userAddress];
                    tempEvents.Add(te);
                    eventsByUserAddress.Add(userAddress, tempEvents);
                }
                else
                {	//resource is not yet in map
                    List<TrendEvent> tempEvents = new List<TrendEvent>();
                    tempEvents.Add(te);
                    eventsByUserAddress.Add(userAddress, tempEvents);
                }
            }

            return eventsByUserAddress;
        }

        private void InitializeEventsIfNecessary()
        {
            if (events == null)
            {	//only happens once
                events = new List<TrendEvent>();
            }
        }

        // singleton code below
        public static InMemoryTrendDataStore GetInstance()
        {
            return InMemoryTrendDataStoreHolder.instance;
        }

        private InMemoryTrendDataStore()
        {
        }

        private static class InMemoryTrendDataStoreHolder
        {
            internal static readonly InMemoryTrendDataStore instance = new InMemoryTrendDataStore();
        }
    }
}